#ifndef V3D_TOOLS_BOUNDOBB_H_INCLUDED
#define V3D_TOOLS_BOUNDOBB_H_INCLUDED

#include "Vector.h"
#include "MatrixTransf.h"
#include "Triangle.h"
#include "BoundAabb.h"

namespace V3D {

//////////////////////////////////////////////////////////////////////////////////////////
// Classe de base des Obb (Oriented Bounding Box)

class BoundObb
{
public:
	// Construction / destruction
	explicit BoundObb( const BoundAabb& aabb);
	explicit BoundObb( const Vector3D& vecInitPoint );
	BoundObb() {}
	~BoundObb() {}

	void Build(const Vect3DArray& aPointCloud, const TriangleArray& aTriangles);
	void Build(const Vect3DArray& aPointCloud);

	inline void ComputeExtremesAlongAxis( const Vector3D& vcAxisOrig, const Vector3f& vcNormedAxis, LengthType& lMin, LengthType& lMax) const;

	void ComputeApproxBoundSphere( Vector3D& vcCenter, float& fRadius) const;

	void TransformBy( const MatrixTransf& mtx );

	const Vector3D& GetCenter() const {return m_matCoordSystem.GetTranslation();}
	const Vector3f& GetAxisX() const {return m_matCoordSystem.GetColumnX();}
	const Vector3f& GetAxisY() const {return m_matCoordSystem.GetColumnY();}
	const Vector3f& GetAxisZ() const {return m_matCoordSystem.GetColumnZ();}
	const Vector3D& GetHalfSizes() const {return m_vcHalfSize;}

	void GetCorners( Vector3D aObbVerts[8]) const;

	inline float GetVolume() const;

private:
	MatrixTransf   m_matCoordSystem;      // Position et orientation du centre de la boite
	Vector3D       m_vcHalfSize;
};

//////////////////////////////////////////////////////////////////////////////////////////


inline BoundObb::BoundObb( const BoundAabb& aabb)
: m_vcHalfSize( aabb.GetHalfSize() )
{
	const Vector3D& vcCenter = aabb.GetCenter();

	m_matCoordSystem.LoadIdentity();
	m_matCoordSystem.SetTranslation( vcCenter);
}

							///////////////////////////////////////////

inline float BoundObb::GetVolume() const
{
	return float( 8.f * m_vcHalfSize.x * m_vcHalfSize.y * m_vcHalfSize.z );
}





inline void BoundObb::ComputeExtremesAlongAxis( const Vector3D& vcAxisOrig, const Vector3f& vcNormedAxis, LengthType& lMin, LengthType& lMax) const
{
	const Vector3f& vcColX = m_matCoordSystem.GetColumnX();
	const Vector3f& vcColY = m_matCoordSystem.GetColumnY();
	const Vector3f& vcColZ = m_matCoordSystem.GetColumnZ();
	Vector3D vcTransl = m_matCoordSystem.GetTranslation();
	vcTransl -= vcAxisOrig;

	LengthType lDist =  LengthType( fabs( vcNormedAxis * vcColX ) * m_vcHalfSize.x
	                              + fabs( vcNormedAxis * vcColY ) * m_vcHalfSize.y
	                              + fabs( vcNormedAxis * vcColZ ) * m_vcHalfSize.z);

	LengthType lMid  =  vcTransl * vcNormedAxis;
	
	lMin = lMid - lDist;
	lMax = lMid + lDist;
	assert( lMin <= lMax );
}






} // namespaces



#endif	// #ifndef V3D_TOOLS_BOUNDOBB_H_INCLUDED


